/**
  Copyright (c) 2010 Freescale Semiconductor
  
  \file  	  Flib1.c
  \brief	  This is the Flash Library Auxiliar #1 File
  \brief	  Provides furhter functionality to animate graphics
  \author	  Freescale Semiconductor
  \author	  Automotive Systems Solutions Engineering
  \author	  IM, b06623
  \version	  2.0
  \revision	  $Revision: 137 $
  \date  	  $Date: 2011-01-11 12:49:40 -0600 (Tue, 11 Jan 2011) $  

* History:  5/Jan/2009 - Initial Version 
*           11/May/2009 - Last changes applied
*			11/Ago/2010	- Added Size animation.
*			10/Sep/2010 - If interpolation is Table, takes values out of the table.


* MISRA VIOLATIONS:
	- [ MISRA 11.2 ]
	- [ MISRA 16.9 ]

* Copyright (c) 2010, Freescale, Inc.  All rights reserved.
*
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
  
*/

#include "FlibInternal.h"
#include "Display.h"

/**
* \brief	FLIB_AnimSetup - Sets up an animation sequence. Animation setup
* \brief	should be done in the callback function registered with the
* \brief	FLIB_Setup call. In cases when multiply animation parameter
* \brief	sets contain errors only one error is indicated by  the return
* \brief	parameter.
* \author	IM, b06623
* \param	sprite: Graphic Object where the animation is performed.
* \param	duration: Number of frames the animation takes. 
* \param	pos: pointer to position animation description.
* \param	alpha: pointer to alpha animation description.
* \param	color: pointer to color animation description
* \param	scrolling: pointer to scolling animation description
* \param	scaling: pointer to scaling animation description
* \param	object: pointer to object animation description
* \return	FLIB_Error_t: FLIB_ERROR_OK (service was accepted),
* \return	FLIB_ERROR_ANIM_POS (Error in alpha animation parameters)
* \return	FLIB_ERROR_ANIM_COLOR (Error in color animation parameters)
* \return	FLIB_ERROR_ANIM_SCROLLING (Error in scrolling animation parameters)
* \return	FLIB_ERROR_ANIM_OBJECT (Error in object animation parameters)
* \todo
*/
FLIB_Error_t FLIB_AnimSetup
(
	uint8_t Layer,
	uint32_t duration,
	const Graphics_Object_t *gobject,
	const FLIB_AnimPos_t* pos,
	const FLIB_AnimAlpha_t* alpha,
	const FLIB_AnimColor_t* color,
	const FLIB_AnimScrolling_t* scrolling,
	const FLIB_AnimScaling_t* scaling,
	const FLIB_AnimObject_t* object
)

{
    FLIB_Error_t error;
    int32_t x_fix;
    int32_t y_fix;

    error = FLIB_ERROR_OK;

    if(FLIB_AnimationData[Layer].duration != 0)
    {
		return FLIB_ERROR_LAYER_ACTIVE;
    }
    if(duration < 2)
    {
		return FLIB_ERROR_PARAM;
    }
    
		
	FLIB_AnimationData[Layer].duration = duration;
	FLIB_AnimationData[Layer].currentFrame = 0;
	
	if(gobject != NULL_PTR)
	{
	    Graphics_CopyGraphicObject(gobject, &FLIB_AnimationData[Layer].gobject);
	}
	else
	{	    
	    FLIB_AnimationData[Layer].gobject.address = (uint32_t)NULL_PTR;
	}

	/* VIOLATION TO [ MISRA 11.2 ] Rationale: Is the only way to indicate that the pointer value is null */
	/* Violation applies to the following 7 lines of C code */      
	FLIB_AnimationData[Layer].anim.pos = NULL_PTR;
	FLIB_AnimationData[Layer].anim.alpha = NULL_PTR;
	FLIB_AnimationData[Layer].anim.color = NULL_PTR;
	FLIB_AnimationData[Layer].anim.scrolling = NULL_PTR;
	FLIB_AnimationData[Layer].anim.scaling = NULL_PTR;
	FLIB_AnimationData[Layer].anim.object = NULL_PTR;
	/* VIOLATION TO [ MISRA 16.9 ] Rationale: Is the only way to indicate that the function pointer is null (used as callback) */  
	FLIB_AnimationData[Layer].callback = NULL_PTR;

	if(pos != NULL_PTR)
	{	    
	    FLIB_AnimationData[Layer].anim.pos = pos;
	
		if(pos->transition != FLIB_TABLE)
		{	
			FLIB_AnimationData[Layer].state.x = ((int32_t)pos->start.x)*65536;
	    	FLIB_AnimationData[Layer].state.y = ((int32_t)pos->start.y)*65536;
	    	
						
			/* calculation of delta increments for each variable */	
			x_fix = ((int32_t)(pos->end.x - pos->start.x))*65536;
			y_fix = ((int32_t)(pos->end.y - pos->start.y))*65536;

			x_fix = x_fix/((int32_t)duration - 1);
			y_fix = y_fix/((int32_t)duration - 1);	


			/* In case of dynamic animation, there is an steady speed state */
			if(FLIB_AnimationData[Layer].anim.pos->transition == FLIB_DYNAMIC)
			{
			    FLIB_AnimationData[Layer].state.dx	= x_fix + (x_fix/2);	    
			    FLIB_AnimationData[Layer].state.dy 	= y_fix + (y_fix/2);
			}
			else
			{	
			    FLIB_AnimationData[Layer].state.dx = x_fix;
			    FLIB_AnimationData[Layer].state.dy = y_fix;
			}
		}
		/* Table animation init values */
		else
		{	
			FLIB_AnimationData[Layer].state.x = ((int32_t)pos->inter_Table[0].x)*65536;
			FLIB_AnimationData[Layer].state.y = ((int32_t)pos->inter_Table[0].y)*65536;
		}	    
	}
	if(alpha != NULL_PTR)
	{
		FLIB_AnimationData[Layer].anim.alpha = alpha;
		
		if(alpha->transition != FLIB_TABLE)
		{ 
			FLIB_AnimationData[Layer].state.alpha = alpha->start*65536;    
			x_fix = ((int32_t)(alpha->end - alpha->start))*65536;
		    x_fix = x_fix/((int32_t)duration - 1);
		    
		    /* In case of dynamic animation, there is an steady speed state */
		    if(FLIB_AnimationData[Layer].anim.alpha->transition == FLIB_DYNAMIC)
		    {
				FLIB_AnimationData[Layer].state.dalpha = x_fix + (x_fix/2);
		    }
		    else
		    {	
				FLIB_AnimationData[Layer].state.dalpha = x_fix;
		    }
	    }
	    else
	    {
	    	FLIB_AnimationData[Layer].state.alpha = ((int32_t)alpha->inter_Table[0])*65536;  
	    }
	}	
	
	
	if(color != NULL_PTR)
	{	
	    FLIB_AnimationData[Layer].anim.color = color;
	}
	
	if(scrolling != NULL_PTR)
	{
	
	    FLIB_AnimationData[Layer].anim.scrolling = scrolling;
	
		if(scrolling->transition != FLIB_TABLE)
		{	
			FLIB_AnimationData[Layer].state.scrx = (int32_t)scrolling->start.x*65536;
	    	FLIB_AnimationData[Layer].state.scry = (int32_t)scrolling->start.y*65536;
	    							
			/* calculation of delta increments for each variable */	
			x_fix = ((int32_t)(scrolling->end.x - scrolling->start.x))*65536;
			y_fix = ((int32_t)(scrolling->end.y - scrolling->start.y))*65536;

			x_fix = x_fix/((int32_t)duration - 1);
			y_fix = y_fix/((int32_t)duration - 1);	


			/* In case of dynamic animation, there is an steady speed state */
			if(scrolling->transition == FLIB_DYNAMIC)
			{
			    FLIB_AnimationData[Layer].state.dscrx	= x_fix + (x_fix/2);	    
			    FLIB_AnimationData[Layer].state.dscry 	= y_fix + (y_fix/2);
			}
			else
			{	
			    FLIB_AnimationData[Layer].state.dscrx = x_fix;
			    FLIB_AnimationData[Layer].state.dscry = y_fix;
			}
		}
		/* Table animation init values */
		else
		{	
			FLIB_AnimationData[Layer].state.scrx = ((int32_t)scrolling->inter_Table[0].x)*65536;
			FLIB_AnimationData[Layer].state.scry = ((int32_t)scrolling->inter_Table[0].y)*65536;
		}


	}
	
	if(scaling != NULL_PTR)
	{		
	
		FLIB_AnimationData[Layer].anim.scaling = scaling;
			
		if(scaling->transition != FLIB_TABLE)
		{ 
			FLIB_AnimationData[Layer].state.scaling = scaling->start*65536;    
			x_fix = ((int32_t)(scaling->end - scaling->start))*65536;
		    x_fix = x_fix/((int32_t)duration - 1);
		    
		    /* In case of dynamic animation, there is an steady speed state */
		    if(scaling->transition == FLIB_DYNAMIC)
		    {
				FLIB_AnimationData[Layer].state.dscaling = x_fix + (x_fix/2);
		    }
		    else
		    {	
				FLIB_AnimationData[Layer].state.dscaling = x_fix;
		    }
	    }
	    else
	    {
	    	FLIB_AnimationData[Layer].state.scaling = ((int32_t)scaling->inter_Table[0])*65536;  
	    }	     
	}
	
	if(object != NULL_PTR)
	{
	    FLIB_AnimationData[Layer].anim.object = object;    
	}


    return FLIB_ERROR_OK;
}
 
 
 
/**
* \brief	FLIB_AnimSetupSize - Alpha version
*/ 
FLIB_Error_t FLIB_AnimSetupSize
(
	uint8_t Layer,
	uint32_t duration,
	const Graphics_Object_t *gobject,
	const FLIB_AnimSize_t* size,
	const FLIB_AnimAlpha_t* alpha,
	const FLIB_AnimColor_t* color
)
{
	int32_t x_fix;
    int32_t y_fix;
    int32_t w_fix;
    int32_t h_fix;
    
    /* Parameter and error checking */
    if(FLIB_AnimationData[Layer].duration != 0)
    {
		return FLIB_ERROR_LAYER_ACTIVE;
    }
    if(duration < 2)
    {
		return FLIB_ERROR_PARAM;
    }
    if(size == NULL_PTR)
    {
		return FLIB_ERROR_ANIM_SIZE;
    }    
    /* end of checking ***************/
	
	FLIB_AnimationData[Layer].duration = duration;
	FLIB_AnimationData[Layer].currentFrame = 0;

	if(gobject != NULL_PTR)
	{
	    Graphics_CopyGraphicObject(gobject, &FLIB_AnimationData[Layer].gobject);
	}
	else
	{
	    FLIB_AnimationData[Layer].gobject.address = (uint32_t)NULL_PTR;
	}
	
	FLIB_AnimationData[Layer].anim.size = size;
	if(FLIB_AnimationData[Layer].anim.size->transition != FLIB_TABLE)
	{	
		FLIB_AnimationData[Layer].state.x = ((int32_t)size->start.x)*65536;
		FLIB_AnimationData[Layer].state.y = ((int32_t)size->start.y)*65536;
		
		FLIB_AnimationData[Layer].state.scrx = ((int32_t)size->start.w)*65536;
		FLIB_AnimationData[Layer].state.scry = ((int32_t)size->start.h)*65536;
		
		/* calculation of delta increments for each variable */	
		x_fix = ((int32_t)(size->end.x - size->start.x))*65536;
		y_fix = ((int32_t)(size->end.y - size->start.y))*65536;
		w_fix = ((int32_t)(size->end.w - size->start.w))*65536;
		h_fix = ((int32_t)(size->end.h - size->start.h))*65536;

		x_fix = x_fix/((int32_t)duration - 1);
		y_fix = y_fix/((int32_t)duration - 1);	
		w_fix = w_fix/((int32_t)duration - 1);
		h_fix = h_fix/((int32_t)duration - 1);

		/* In case of dynamic animation, there is an steady speed state */
		if(FLIB_AnimationData[Layer].anim.size->transition == FLIB_DYNAMIC)
		{
		    FLIB_AnimationData[Layer].state.dx 		= x_fix + (x_fix/2);	    
		    FLIB_AnimationData[Layer].state.dy  	= y_fix + (y_fix/2);
		    FLIB_AnimationData[Layer].state.dscrx 	= w_fix + (w_fix/2);    
		    FLIB_AnimationData[Layer].state.dscry  	= h_fix + (h_fix/2);
		}
		else
		{	
		    FLIB_AnimationData[Layer].state.dx = x_fix;
		    FLIB_AnimationData[Layer].state.dy = y_fix;
		    FLIB_AnimationData[Layer].state.dscrx = w_fix;
		    FLIB_AnimationData[Layer].state.dscry = h_fix;    
		}
	}
	/* Table animation init values */
	else
	{	
		FLIB_AnimationData[Layer].state.x = ((int32_t)size->inter_Table[0].x)*65536;
		FLIB_AnimationData[Layer].state.y = ((int32_t)size->inter_Table[0].y)*65536;
		
		FLIB_AnimationData[Layer].state.scrx = ((int32_t)size->inter_Table[0].w)*65536;
		FLIB_AnimationData[Layer].state.scry = ((int32_t)size->inter_Table[0].h)*65536;
	}	
		
	/* VIOLATION TO [ MISRA 11.2 ] Rationale: Is the only way to indicate that the pointer value is null */
	/* Violation applies to the following 6 lines of C code */
	if(alpha != NULL_PTR)
	{
		FLIB_AnimationData[Layer].anim.alpha = alpha;
		
		if(FLIB_AnimationData[Layer].anim.size->transition != FLIB_TABLE)
		{ 
			FLIB_AnimationData[Layer].state.alpha = alpha->start*65536;    
			x_fix = ((int32_t)(alpha->end - alpha->start))*65536u;
		    x_fix = x_fix/((int32_t)duration - 1);
		    /* In case of dynamic animation, there is an steady speed state */
		    if(FLIB_AnimationData[Layer].anim.alpha->transition == FLIB_DYNAMIC)
		    {
				FLIB_AnimationData[Layer].state.dalpha = x_fix + (x_fix/2);
		    }
		    else
		    {	
				FLIB_AnimationData[Layer].state.dalpha = x_fix;
		    }
	    }
	    else
	    {
	    	FLIB_AnimationData[Layer].state.alpha = ((int32_t)alpha->inter_Table[0])*65536;  
	    }
	}
	
	if(color != NULL_PTR)
	{     
		FLIB_AnimationData[Layer].anim.color = color;
	}
	
	/* VIOLATION TO [ MISRA 11.2 ] Rationale: Is the only way to indicate that the pointer value is null */
	FLIB_AnimationData[Layer].anim.scrolling 	= NULL_PTR;
	FLIB_AnimationData[Layer].anim.scaling 		= NULL_PTR;
	FLIB_AnimationData[Layer].anim.object 		= NULL_PTR;
	/* VIOLATION TO [ MISRA 16.9 ] Rationale: Is the only way to indicate that the function pointer is null (used as callback) */  
	FLIB_AnimationData[Layer].callback 			= NULL_PTR;
	
    return FLIB_ERROR_OK;
}